/**
* \file: AditVideoSink.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*          Y.Nakamura/ ADITJ    /  ynakamura@jp.adit-jv.com
* 
* \copyright (c) 2014-2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_ADITVIDEOSINK_H
#define AAUTO_ADITVIDEOSINK_H

#include <aauto/VideoSink.h>
#include <aauto/MessageRouter.h>
#include "aauto/AautoErrorCodes.h"

namespace adit { namespace aauto {

/**
 * @brief callback function to receive notification from MD.
 * All callbacks have empty default implementations.
 * If necessary, need to be implemented . */
class IAditVideoSinkCallbacks
{
public:
    virtual ~IAditVideoSinkCallbacks() {}
    
    /**
     * @brief called when setup message is received.
     * Signals that the other side wants to set up a video stream
     * and the information in this callback should be used
     * to set up any hardware necessary to handle the incoming stream type.
     * @param inMediaCodecType The media codec type requested (Currently, the only valid value is MEDIA_CODEC_VIDEO_H264_BP).*/
    virtual void setupCallback(int inMediaCodecType) {}

    /**
     * @brief called when playback is goingt to start. 
     * This is invoked after only the video environment setup has finished*/ 
    virtual void playbackStartCallback() {}
    
    /**
     * @brief called when first frame is ready.
     * This is invoked after the first frame is rendered*/
    virtual void playbackFirstFrameRenderedCallback() {}

    /**
     * @brief called when playback has ended. 
     * This is invoked before the video environment is cloesed.
     * This is invoked after outputting all data from GalReceiver to video environment.
     * Therefore, video data are not left in AditVideoSink.*/
    virtual void playbackStopCallback() {}
    
    /**
     * @brief called when source sends video configuration. This configuration can be
     * different from what is originally set by setConfigItem. 
     * @param inVideoWidth The width of codec resolution selected.
     * @param inVideoHeight The height of codec resolution seletected.
     * @param inUIResWidth The width of the logical UI resolution
     * @param inUIResHeight The height of the logical UI resolution */
    virtual void sourceVideoConfigCallback(int inVideoWidth, int inVideoHeight, 
            int inUIResWidth, int inUIResHeight) {(void)inVideoWidth; (void)inVideoHeight; (void)inUIResWidth; (void)inUIResHeight;}
    
    /**
     * @brief called when the MD wishes to change video focus. you must call setVideoFocus(int focusMode, bool unsolicited)
     *         with appropriate focus mode to deny or grant the request. If you
     *         want to grant a focus request, set the same focus as was requested.
     *         To deny a request, just set the video focus request back to whatever
     *         it already was. focusMode can be VIDEO_FOCUS_PROJECTED or VIDEO_FOCUS_NATIVE.
     *         unsolicited will be true if responding to a videoFocusCallback, false otherwise.
     * @param inFocus The focus mode request by the other end. Can be one of VIDEO_FOCUS_PROJECTED or VIDEO_FOCUS_NATIVE
     * @param inReason The reason for this video focus request. One of VideoFocusReason. */
    virtual void videoFocusCallback(int inFocus, int inReason) {(void)inFocus; (void)inReason;}

    /** @brief Called when an error happens in the underlying layer.
    * The callback is introduced to inform the upper layer about error cases,
    * for which another notification, such as return values, is not possible.
    * @param inErrorCode  Error code
    * */
    virtual void notifyErrorCallback(aautoErrorCodes inErrorCode) {(void) inErrorCode;}
};

//TODO how to display not-shadowed methods from VideoSink in doxygen.
/** 
 * @brief AditVideoSink provides a generic base class to video implementations provided by ADIT.
 * It hides some VideoSink methods and replaces them by setConfigItem. */
class AditVideoSink : public VideoSink
{
public:

/**
 * @brief  Constructor of AditVideoSink
 * @param  inSessionId      inSessionId is Id that is used in GalReceiver as identification data on session.
 * @param  inMessageRouter    inMessageRouter is message router obtained from GalReceiver
 * @param  inAutoStart      true if starting projection mode automatically, otherwise false. */
    AditVideoSink(uint8_t inSessionId, MessageRouter* inMessageRouter, bool inAutoStart) :
        VideoSink(inSessionId, inMessageRouter, inAutoStart) {}
    
    /**
     * @brief Set configuration for platform and AAuto-related items.
     *         All configurations must be called before init().
     * @param  inKey        The key string for configuration
     *      - video-codec [optional, default="MEDIA_CODEC_VIDEO_H264_BP"]: The media codec type which needs to be set. In galreceiver1.0 only "MEDIA_CODEC_VIDEO_H264_BP" is available.
     *      - video-max-unacked-frames [optional, default=4]: How many frames should be kept outstanding.
     *      - video-width [mandatory]: The number of pixels in the x axis for a logical UI resolution. (800 < video-width <= 1920)
     *      - video-height [mandatory]: The number of pixels in the y axis for a logical UI resolution. (480 < video-height <= 1080)
     *          To ensure support for a wide range of physical in-vehicle display hardware, AAP uses a logical UI resolution that is
     *          always less or equal to the codec resolution of the transported video stream.
     *          Currently codec resolution of the video stream from the mobile device
     *          can be 800x480 or 1280x720 or 1920x1080. This will be calculated internally according to input logical UI resolution. 
     *          Horizontal and vertical margin will be filled with an arbitrary color if UI resolution is less than codec resolution.
     *          Please see an integration manual for more detail.
     *      - video-density [mandatory]: The density in dpi.(0 < video-density)
     *      - video-fps [mandatory]: Frame rate of video. (must be "30" or "60")
     *      - video-additional-depth [optional]: Additional frames decoder needs to display a frame. (default value 0)
     * @param  inValue      The actual value for configuration */     
    virtual void setConfigItem(string inKey, string inValue) = 0;
    
    /**
     * @brief Register callback interface to receive video-related notifications.
     * See IAditVideoSinkCallbacks for details.
     *
     * @param   inCallbacks object implementing the callback interface */
    virtual void registerCallbacks(IAditVideoSinkCallbacks* inCallbacks) = 0;
    
    // TODO error code to be more detailed than false?
    /**
     * @brief Initialize the video sink. All configuration items must be set.
     *
     * @return true if initialization succeeds.
     *          false if configuration items without default value is not set or initialization
     *          fails. */
    virtual bool init() = 0;

    /**
     * @brief  Shut down the video sink, stop running threads and release allocated resources */
    virtual void shutdown() = 0;

protected:
    /**
     * @brief Method removed from public interface. Use setConfigItem instead. */
    void setMaxUnackedFrames(int maxUnackedFrams);
    
    /**
     * @brief Method removed from public interface. Use setConfigItem instead. */
    uint32_t addSupportedConfiguration(int codecResolution, int32_t widthMargin, int32_t heightMargin, int32_t density);
};

} } /* namespace adit { namespace aauto { */

#endif /* AAUTO_ADITVIDEOSINK_H */
